package gov.va.med.domain.service.messaging.parse;

import gov.va.med.domain.service.messaging.MessagingConstants;
import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.DataTypeException;
import ca.uhn.hl7v2.model.Message;
import ca.uhn.hl7v2.model.Primitive;
import ca.uhn.hl7v2.model.Segment;
import ca.uhn.hl7v2.model.Structure;
import ca.uhn.hl7v2.model.Type;
import ca.uhn.hl7v2.model.Varies;
import ca.uhn.hl7v2.parser.EncodingCharacters;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;
import ca.uhn.hl7v2.parser.Escape;
import ca.uhn.hl7v2.parser.Parser;
import ca.uhn.hl7v2.parser.PipeParser;
import ca.uhn.hl7v2.util.FilterIterator;
import ca.uhn.hl7v2.util.MessageIterator;
import ca.uhn.hl7v2.util.Terser;
import ca.uhn.log.HapiLog;
import ca.uhn.log.HapiLogFactory;

/**
 * Parses HL7 message selectively and validates against the expected HL7 Version. 
 * Key extensions:
 * <UL>
 * <LI>IHL7ParseEventListener is used to determine which segments should be parsed.  Required  
 * <LI>IHL7ParseErrorListener is used to report feild and segment errors to allow parsing 
 * to continue despite low level parsing errors. Required
 * <LI>IHL7MessageClassOverrideListener is used to allow alternate message strucutres otherwise
 * unsupported by HAPI.  Optional
 * </UL> 
 * 
 *
 * @author Slava Uchitel
 * @version $Id: HL7FilteringParser.java,v 1.15 2005/09/09 21:08:47 tom Exp $
 * @since MHV 2.0 <br>03/16/2005
 */

public class HL7FilteringParser extends PipeParser { 
  
	private IHL7ParseEventListener eventListener;
	private IHL7ParseErrorListener errorListener;
	private IHL7MessageClassOverrideListener classOverrideListener;
	
	private String expectedHL7Version = MessagingConstants.DEFAULT_EXPECTED_HL7_VERSION; 
	private final static String segDelim = "\r";
	private static final HapiLog log = HapiLogFactory.getHapiLog(HL7FilteringParser.class);

	public HL7FilteringParser() {
		super();
	}
	public HL7FilteringParser(String anHL7Version) {
		super();
		setExpectedHL7Version(anHL7Version);
		
	}

	/**
	 * Returns object that contains the field separator and encoding characters
	 * for this message.
	 */
	private static EncodingCharacters getEncodingChars(String message) {
		return new EncodingCharacters(message.charAt(3), message.substring(4, 8));
	}
	
	/** 
     * @return true if the segment is MSH, FHS, or BHS.  These need special treatment 
     *  because they define delimiters.
     * @param theSegmentName
     */
    private static boolean isDelimDefSegment(String theSegmentName) {
         
        return theSegmentName.equals("MSH") 
            	|| theSegmentName.equals("FHS") 
            	|| theSegmentName.equals("BHS");
 
    }
    
	/** 
     * @return true if Parse Exceptions for desination segment and field number are to be ignored.
     * These are fields that the messaging framework does not care about when building DTOs.
     * @param destination - segment
     * @param fieldNumber
     */
    private static boolean isIgnorableParseFieldError(String destination, int fieldNumber) {
         if (destination.equalsIgnoreCase("OBR") && (fieldNumber == 32)) {
         	return true;
         }
         if (destination.equalsIgnoreCase("OBR") && (fieldNumber == 33)) {
         	return true;
         }
         if (destination.equalsIgnoreCase("OBR") && (fieldNumber == 34)) {
         	return true;
         }
         if (destination.equalsIgnoreCase("OBR") && (fieldNumber == 35)) {
         	return true;
         }
        return false;
    }
    
	/**
	 * This method is a complete override of the HAPI PipeParser.parse  method solely 
     * to:
     * <UL>
     * <LI> Place a try/catch around the HL7Exception for segment errors and report it to the 
     * error listener.   
     * <LI> Call our own parseField method which is necessary because HAPI PipeParse made its
     *  version static and private. 
     * <LI> Ask the IHL7MessageClassOverrideListener if there are any overrides to allow alternate
     *  Message Struture (e.g. adding QAK segment otherwise unsupported by HAPI)
     * </UL>.
	 *
	 * @throws HL7Exception if the message is not correctly formatted.
	 * @throws EncodingNotSupportedException if the message encoded
	 * is not supported by this parser.
	 */
	public synchronized Message parse(String message) throws HL7Exception, EncodingNotSupportedException {

		String version = this.getVersion(message);
	    validate(message, version);
	    
		Message m = null;
		String structure = getMessageStructure(message);
		
		//get list of packages to search for the corresponding message class
		String[] packages = Parser.packageList(version);

		//try to instantiate a message object of the right class
		try {
			Class messageClass = Parser.findMessageClass(structure, version, false);
			// Begin MHV Override to allow unsupported HAPI segemnts to be processed e.g. QAK segment
			if (this.getClassOverrideListener() != null)
			{    
			    Class overrideClass = getClassOverrideListener().onFindMessageClass(messageClass, structure, version);
			    messageClass=(overrideClass != null?overrideClass:messageClass);
			}
			// End Mhv Override.  
			if(messageClass == null)
				throw new ClassNotFoundException("Can't find message class in current package list: " + structure);
			log.info("Parsing message of class " + messageClass.getName());
			m = (Message)messageClass.newInstance();
		}
		catch(Exception e) {
			StringBuffer pckgs = new StringBuffer();
			for(int i = 0; i < packages.length; i++) {
				pckgs.append(packages[i]);
				pckgs.append("  ");
			}

			throw new HL7Exception("Couldn't create Message object of type " + structure,
			                       HL7Exception.UNSUPPORTED_MESSAGE_TYPE,
			                       e);
		}

		//MessagePointer ptr = new MessagePointer(this, m, getEncodingChars(message));
		MessageIterator messageIter = new MessageIterator(m, "MSH", true);
		FilterIterator.Predicate segmentsOnly = new FilterIterator.Predicate() {
			public boolean evaluate(Object obj) {
				return Segment.class.isAssignableFrom(obj.getClass());
			}
		};
		FilterIterator segmentIter = new FilterIterator(messageIter, segmentsOnly);

		String[] segments = split(message, segDelim);
		for(int i = 0; i < segments.length; i++) {

			//get rid of any leading whitespace characters ...
			if(segments[i] != null && segments[i].length() > 0 && Character.isWhitespace(segments[i].charAt(0)))
				segments[i] = stripLeadingWhitespace(segments[i]);

			//sometimes people put extra segment delimiters at end of message ...
			if(segments[i] != null && segments[i].length() >= 3) {
				final String name = segments[i].substring(0, 3);
				if(getEventListener().onSegmentStart(name)) {
					log.debug("Parsing segment " + name);
				}
				else {
					log.debug("Skipping segment " + name);
					continue;
				}
				messageIter.setDirection(name);
				FilterIterator.Predicate byDirection = new FilterIterator.Predicate() {
					public boolean evaluate(Object obj) {
						return ((Structure)obj).getName().equals(name);
					}
				};
				FilterIterator dirIter = new FilterIterator(segmentIter, byDirection);
				if(dirIter.hasNext()) {
					Segment seg = (Segment)dirIter.next();
					try
					{
					    parseSegment(seg, segments[i], getEncodingChars(message));
					}
					catch (HL7Exception e)
					{
					    HL7ParseSegmentError error = new HL7ParseSegmentError();
					    error.setStructure(seg);
					    error.setStrSeg(segments[i]);
					    error.setException(e);
					    getErrorListener().onParseSegmentException(error);
					}
					getEventListener().onSegmentEnd(name, seg);
				}
			}
		}
		return m;
	}
    /**
     * This method is a complete replciation of the HAPI PiepParser.parseField method solely 
     * to place a try/catch around the DataTypeException and report it to the 
     * error listener.  
     */
    protected void parseField(Segment destination, 
            				  Type destinationField, 
            				  int repetitionNumber, 
            				  String data, 
            				  EncodingCharacters encodingCharacters,
							  int fieldNumber) 
                  throws HL7Exception {
        
        String[] components = split(data, String.valueOf(encodingCharacters.getComponentSeparator()));
        for (int i = 0; i < components.length; i++) {
            String[] subcomponents = split(components[i], String.valueOf(encodingCharacters.getSubcomponentSeparator()));
            for (int j = 0; j < subcomponents.length; j++) {
                String val = subcomponents[j];
                if (val != null) {
                    val = Escape.unescape(val, encodingCharacters);
                }
                Primitive primitive = Terser.getPrimitive(destinationField, i+1, j+1);
                try
                {
                    primitive.setValue(val);                
                }
                catch (DataTypeException e)
                {
                	//FIXME This is temporary work-around until Dirty Data is fully implemented.
                	//Contains hard-coded references to segments and fields decoders don't need.
                	if (isIgnorableParseFieldError(destination.getName(), fieldNumber)) {
                		continue;
                	}
                    HL7ParseFieldError error = new HL7ParseFieldError();
                    error.setStructure(destination);
                    error.setFieldNumber(fieldNumber);
                    error.setRepetitionNumber(repetitionNumber);
                    error.setComponentNumber(j);
                    error.setDestinationField(destinationField);
                    error.setData(data);
                    error.setPrimitive(primitive);
                    error.setValue(val);
                    error.setException(e);
                    getErrorListener().onParseFieldException(error);
                }
            }
        }
    }
    
    /**
     * Parses a segment string and populates the given Segment object.  Unexpected fields are
     * added as Varies' at the end of the segment.  
     *
     * @throws HL7Exception if the given string does not contain the
     *      given segment or if the string is not encoded properly
     */
	public void parseSegment(Segment destination, String segment, EncodingCharacters encodingChars) throws HL7Exception {
        int fieldOffset = 0;
        if (isDelimDefSegment(destination.getName())) {
            fieldOffset = 1;
            //set field 1 to fourth character of string
            Terser.set(destination, 1, 0, 1, 1, String.valueOf(encodingChars.getFieldSeparator()));
        }
        
        String[] fields = split(segment, String.valueOf(encodingChars.getFieldSeparator()));
        //destination.setName(fields[0]);
        for (int i = 1; i < fields.length; i++) {
            String[] reps = split(fields[i], String.valueOf(encodingChars.getRepetitionSeparator()));
            log.debug(reps.length + " reps delimited by: " + encodingChars.getRepetitionSeparator());
            
            //MSH-2 will get split incorrectly so we have to fudge it ...
            boolean isMSH2 = isDelimDefSegment(destination.getName()) && i+fieldOffset == 2;
            if (isMSH2) {  
                reps = new String[1];
                reps[0] = fields[i];
            }
            
            for (int j = 0; j < reps.length; j++) {
                try {
                    StringBuffer statusMessage = new StringBuffer("Parsing field ");
                    statusMessage.append(i+fieldOffset);
                    statusMessage.append(" repetition ");
                    statusMessage.append(j);
                    log.debug(statusMessage.toString());
                    //parse(destination.getField(i + fieldOffset, j), reps[j], encodingChars, false);

                    Type field = destination.getField(i + fieldOffset, j);
                    if (isMSH2) {
                        Terser.getPrimitive(field, 1, 1).setValue(reps[j]);
                    } else {
                        parseField(destination, field, j, reps[j], encodingChars, i);
                    }
                }
                catch (HL7Exception e) {
                    //set the field location and throw again ...
                    e.setFieldPosition(i);
                    e.setSegmentRepetition(MessageIterator.getIndex(destination.getParent(), destination).rep);
                    e.setSegmentName(destination.getName());
                    throw e;
                }
            }
        }
        
        //set data type of OBX-5
        if (destination.getClass().getName().indexOf("OBX") >= 0) {
            Varies.fixOBX5(destination);
        }
        
    }
	
	protected void validate(String message, String version) throws HL7Exception{
		if(getErrorListener() == null )
			throw new HL7Exception("ErrorListener can not be null.");
		if ( getEventListener() == null)
			throw new HL7Exception("EventListener can not be null.");
		
		if(!version.equalsIgnoreCase(getExpectedHL7Version())) {
			throw new HL7Exception("Can't process message of version '" + version + "' - version not supported",
			                       HL7Exception.UNSUPPORTED_VERSION_ID);
		} 
		if(!validVersion(version)) {
			throw new HL7Exception("Can't process message of version '" + version + "' - version not recognized",
			                       HL7Exception.UNSUPPORTED_VERSION_ID);
		}
	}
    public String getExpectedHL7Version() {
        return expectedHL7Version;
    }
    public void setExpectedHL7Version(String expectedHL7Version) {
        this.expectedHL7Version = expectedHL7Version;
    }
    public IHL7ParseErrorListener getErrorListener() {
        return errorListener;
    }
    public void setErrorListener(IHL7ParseErrorListener errorListener) {
        this.errorListener = errorListener;
    }
    public IHL7ParseEventListener getEventListener() {
        return eventListener;
    }
    public void setEventListener(IHL7ParseEventListener eventListener) {
        this.eventListener = eventListener;
    }
    /**
     * @return Returns the classOverrideListener.
     */
    public IHL7MessageClassOverrideListener getClassOverrideListener()
    {
        return classOverrideListener;
    }
    /**
     * @param classOverrideListener The classOverrideListener to set.
     */
    public void setClassOverrideListener(
            IHL7MessageClassOverrideListener classOverrideListener)
    {
        this.classOverrideListener = classOverrideListener;
    }
}
